﻿using UnityEngine;

public enum SortingOrigin
{
    Pivot, Bottom, Top, Left, Right
}
public enum SortingAxis
{
    X, Y
}

[RequireComponent(typeof(Renderer))]
public class DepthSorter : MonoBehaviour
{

    [SerializeField]
    private int _OriginOrder = 100;

    [SerializeField]
    private float _Precision = 1f;

    [SerializeField]
    private float _OriginOffset = 0.0f;

    [SerializeField]
    private bool _RunOnlyOnce = false;

    [SerializeField]
    private SortingOrigin _SortingOrigin;

    [SerializeField]
    private SortingAxis _SortingAxis = SortingAxis.Y;

    private Renderer _Renderer;
    private float _RemainingTime;
    private float _UpdateFrequency = 0.1f;

    private void Awake()
    {
        _Renderer = GetComponent<Renderer>();
    }
    private void Start()
    {
        UpdateSortOrder();
        if (_RunOnlyOnce)
        {
            enabled = false;
        }
    }

    private void LateUpdate()
    {
        UpdateSortOrder();
    }

    private void UpdateSortOrder()
    {
        _RemainingTime -= Time.deltaTime;
        if (_RemainingTime <= 0)
        {
            _RemainingTime = _UpdateFrequency;

            Vector2 pos = _Renderer.bounds.center;
            float width = _Renderer.bounds.size.x;
            float height = _Renderer.bounds.size.y;

            switch (_SortingOrigin)
            {
                case SortingOrigin.Bottom:
                    pos += Vector2.down * height / 2;
                    break;

                case SortingOrigin.Top:
                    pos += Vector2.up * height / 2;
                    break;

                case SortingOrigin.Left:
                    pos += Vector2.left * width / 2;
                    break;

                case SortingOrigin.Right:
                    pos += Vector2.right * width / 2;
                    break;

                default:
                    pos = transform.position;
                    break;
            }

            float posFromAxis = _SortingAxis == SortingAxis.X ? pos.x : pos.y;
            _Renderer.sortingOrder = (int)(_OriginOrder - posFromAxis / _Precision + _OriginOffset);
        }
    }
}